import { Image, StyleSheet, Text, TouchableOpacity, View, Linking } from 'react-native';
import React, { useEffect, useRef, useState } from 'react';
import Modal from 'react-native-modal';
import MapView, { Marker, AnimatedRegion } from 'react-native-maps';
import MapViewDirections from 'react-native-maps-directions';
//import Geocoder from 'react-native-geocoding';
import { GOOGLE_MAPS_API_KEY } from '../../config'; 
import Header from '../../components/Header';
import { fp, hp, wp } from '../../helpers/Dimensions'
import { DROPICON, MAN, PHONE, PICKDROPLOCMAP, PICKICON, STAR } from '../../assets/Icon';
import Buttonblack from '../../components/Buttonblack';
import { useNavigation } from '@react-navigation/native';
import RideService, { getInProcessRide } from '../../services/RideService'
import { getDistance } from 'geolib';


const Rideaccept = () => {
    const navigation = useNavigation();
    const [statusMessage, setStatusMessage] = useState(false);
    const [rideInfo, setRideInfo] = useState(null);
    const [isModalVisible, setModalVisible] = useState(false);
    const [region, setRegion] = useState(null);
    const [pickupLocation, setPickupLocation] = useState(null);
    const [currentLocation, setCurrentLocation] = useState(null);
    const [driverLocation, setDriverLocation] = useState(null);
    const [rideAccepted, setRideAccepted] = useState(false);
    const [mLat, setMLat] = useState(null); // Default latitude
    const [mLong, setMLong] = useState(null); // Default longitude
    const trackingInterval = useRef(null);

    useEffect(() => {
        //Geocoder.init(GOOGLE_MAPS_API_KEY);
        acceptRide();
    }, []);


    const requestLocationPermission = async () => {
            try {
              const granted = await PermissionsAndroid.request(
                PermissionsAndroid.PERMISSIONS.ACCESS_FINE_LOCATION,
                {
                  title: 'Location Permission',
                  message:
                    'This app needs access to your location to show your position on the map.',
                  buttonNeutral: 'Ask Me Later',
                  buttonNegative: 'Cancel',
                  buttonPositive: 'OK',
                },
              );
              if (granted === PermissionsAndroid.RESULTS.GRANTED) {
                console.log("Location permission granted");
                getLocation();
              } else {
                console.log('Location permission denied');
                setLoading(false);
              }
            } catch (err) {
              console.warn(err);
              setLoading(false);
            }
          };
    
    const getLocation = () => {
            Geolocation.getCurrentPosition(
              (position) => {
                console.log("your loc: ", position);
                setMLat(position.coords.latitude);
                setMLong(position.coords.longitude);
                setLoading(false);
              },
              (error) => {
                // See error code charts below.
                console.log(error.code, error.message);
                setLoading(false);
              },
              { enableHighAccuracy: true, timeout: 15000, maximumAge: 10000 }
          );
          }


    const acceptRide = async () => {
        try {
            const inProcessRide = await getInProcessRide();
            console.log("rideaccept:== ", inProcessRide);
        
            if (!inProcessRide || inProcessRide.length === 0) return;

            const ride = inProcessRide[0];
            setRideInfo(ride);

            if (ride?.driver?.latitude && ride?.driver?.longitude) {
                const initialDriverLocation = {
                    latitude: parseFloat(ride.driver.latitude),
                    longitude: parseFloat(ride.driver.longitude),
                };
                setDriverLocation(initialDriverLocation);
                setCurrentLocation(initialDriverLocation);
            }

            if (ride?.pickup_latitude && ride?.pickup_longitude) {
                const userPickupLocation = {
                    latitude: parseFloat(ride.pickup_latitude),
                    longitude: parseFloat(ride.pickup_longitude),
                };
                setPickupLocation(userPickupLocation);
            }
        } catch (error) {
            console.error("Error fetching ride data:", error);
        }
    };

    // Start tracking only when both driver & pickup locations are set
    useEffect(() => {
        if (driverLocation && pickupLocation) {
            startRideTracking(driverLocation, pickupLocation);
        }
    }, [driverLocation, pickupLocation]);

    const startRideTracking = (src, dest) => {
        if (!src || !dest) return;

        let progress = 0;
        const distance = getDistance(src, dest); // Calculate distance in meters
        const step = distance > 1000 ? 0.02 : 0.05; // Adjust movement step based on distance

        console.log("Starting Live Tracking...");

        trackingInterval.current = setInterval(() => {
            progress += step;

            const newLat = src.latitude + (dest.latitude - src.latitude) * progress;
            const newLng = src.longitude + (dest.longitude - src.longitude) * progress;
            const newPosition = { latitude: newLat, longitude: newLng };

            console.log("Updating Driver Location:", newPosition);
            setCurrentLocation(newPosition);

            const remainingDistance = getDistance(newPosition, dest);
            console.log("Remaining Distance:", remainingDistance);

            if (remainingDistance <= 50 || progress >= 1) {
                clearInterval(trackingInterval.current);
                setStatusMessage(true);
                checkRideStartStatus();                
            }
        }, 2000);
    };

    const checkRideStartStatus = async () => {
        console.log("Checking ride start status...");
    
        const checkInterval = setInterval(async () => {
            try {
                console.log("rideini==", rideInfo.id);
                const rideStartDtl = await rideStart(rideInfo?.id); 
                console.log("Ride Start Status:", rideStartDtl);
                if(rideStartDtl){
                    clearInterval(checkInterval); 
                    console.log("Ride has started! Navigating to next screen...");
                    navigation.navigate("Trackdelivery"); 
                    
                }
            } catch (error) {
                console.error("Error checking ride start status:", error);
            }
        }, 3000);
    };

    return (
        <View style={styles.mainContainer}>
            <Header title="Parcel" link="Chooserideparcel" />

            {driverLocation && pickupLocation && (
                <MapView
                    style={styles.map}
                    initialRegion={{
                        latitude: driverLocation.latitude,
                        longitude: driverLocation.longitude,
                        latitudeDelta: 0.05,
                        longitudeDelta: 0.05,
                    }}
                >
                    {/* Animated Driver Marker */}
                   
                    <Marker coordinate={currentLocation} title="Driver Location">
                         <Image source={MAN} style={{ width: 40, height: 40 }} />
                     </Marker>
                    

                    {/* User Pickup Location Marker */}
                    <Marker coordinate={pickupLocation} title="Pickup Location" />

                    {/* Polyline from Driver to Pickup Location */}
                    <MapViewDirections
                        origin={driverLocation}
                        destination={pickupLocation}
                        apikey={GOOGLE_MAPS_API_KEY}
                        strokeWidth={4}
                        strokeColor="blue"
                    />
                </MapView>
            )}

            {/* Button to Show Driver Info */}
            <TouchableOpacity
                style={styles.showDriverButton}
                onPress={() => setModalVisible(true)}
            >
                <Text style={styles.buttonText}>View Driver Info</Text>
            </TouchableOpacity>

            {/* Driver Info Modal */}
            <Modal isVisible={isModalVisible} style={styles.modal} onBackdropPress={() => setModalVisible(false)}>
                <View style={styles.modalContent}>
                    <View style={styles.driverInfoContainer}>
                        <Image source={MAN} style={styles.driverImage} />
                        <Text style={styles.textH2}>{rideInfo?.driver?.name || "N/A"}</Text>
                        <View style={styles.starContainer}>
                            <Image source={STAR} />
                            <Image source={STAR} />
                            <Image source={STAR} />
                            <Image source={STAR} />
                            <Image source={STAR} />
                        </View>
                        <TouchableOpacity onPress={() => Linking.openURL(`tel:${rideInfo?.driver?.mobile || ""}`)}>
                            <Image source={PHONE} style={styles.phoneIcon} />
                        </TouchableOpacity>
                        <Text style={styles.textH2Normal}>
                            {rideInfo?.driver?.brand}, {rideInfo?.driver?.model}, {rideInfo?.driver?.vehicle_type || "N/A"}, {rideInfo?.driver?.registration_number}
                        </Text>
                    </View>
                    <Buttonblack title="Close" wide="70" onPress={() => setModalVisible(false)} />
                    
                </View>
            </Modal>
            {statusMessage && (
                <Buttonblack title="Driver Reached your location" wide="70"/>
            )}
            <Buttonblack title="Cancel" wide="70" onPress={() => navigation.navigate("Cancelride")} />
        </View>
    )
}

export default Rideaccept

const styles = StyleSheet.create({
    mainContainer: {
        flex: 1,
        backgroundColor: "#FFFFFF",
    },
    map: {
        flex : 1,
    },
    showDriverButton: {
        backgroundColor: "#000",
        padding: 10,
        margin: 10,
        borderRadius: 5,
        alignItems: "center",
    },
    buttonText: {
        color: "#FFFFFF",
        fontSize: 2,
        fontWeight: "bold",
    },
    modal: {
        justifyContent: "flex-end",
        margin: 0,
    },
    modalContent: {
        backgroundColor: "#FFFFFF",
        padding: 20,
        borderTopLeftRadius: 20,
        borderTopRightRadius: 20,
    },
    driverInfoContainer: {
        alignItems: "center",
    },
    driverImage: {
        width: 60,
        height: 60,
        borderRadius: 30,
        marginBottom: 10,
    },
    textH2: {
        color: "#000000",
        fontSize: 2.5,
        fontWeight: "bold",
        textAlign: "center",
    },
    textH2Normal: {
        color: "#000000",
        fontSize: 2,
        textAlign: "center",
    },
    phoneIcon: {
        width: 30,
        height: 30,
        marginVertical: 10,
    },
    starContainer: {
        flexDirection: "row",
        marginVertical: 5,
    },
    markerIcon: { width: 40, height: 40 },
})
